<?php
/**
 * Mass BIN Check - versi web
 * Baca list dari list.txt (tiap baris: 6 digit BIN atau 16 digit kartu, ambil 6 digit pertama)
 * Hasil ditulis ke result.txt
 */
set_time_limit(0);
ignore_user_abort(true);

$DIR = __DIR__;
$LIST_FILE = $DIR . '/list.txt';
$RESULT_FILE = $DIR . '/result.txt';
$DELAY_SEC = 1; // jeda tiap request (supaya tidak kena rate limit)

function fetch_url($url, $proxyUrl = null) {
    $ch = curl_init($url);
    curl_setopt_array($ch, [
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_FOLLOWLOCATION => true,
        CURLOPT_TIMEOUT        => 12,
        CURLOPT_CONNECTTIMEOUT => 6,
        CURLOPT_SSL_VERIFYPEER => true,
        CURLOPT_USERAGENT      => 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/120.0.0.0 Safari/537.36',
    ]);
    if (!empty($proxyUrl)) {
        curl_setopt($ch, CURLOPT_PROXY, $proxyUrl);
        curl_setopt($ch, CURLOPT_PROXYTYPE, CURLPROXY_SOCKS5);
    }
    $body = curl_exec($ch);
    $err = curl_error($ch);
    $code = (int)curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);
    return ['body' => $body, 'error' => $err, 'http_code' => $code];
}

function is_valid_bincheck_response($html) {
    if ($html === false || $html === '' || strlen($html) < 500) return false;
    return (stripos($html, 'Card Brand') !== false || stripos($html, 'BIN/IIN') !== false || stripos($html, 'valid BIN number') !== false);
}

/** Mengembalikan satu baris hasil: "Bin            :: 519234 MASTERCARD DEBIT ..." atau baris gagal */
function get_bin_line($bin) {
    $bin = substr(preg_replace('/\D/', '', $bin), 0, 6);
    if (strlen($bin) < 6) return null;

    $url = 'https://bincheck.io/details/' . $bin;
    $html = false;

    for ($attempt = 0; $attempt < 2; $attempt++) {
        $res = fetch_url($url, null);
        $html = $res['body'];
        if ($html !== false && is_valid_bincheck_response($html)) break;
        if ($attempt < 1) usleep(300000);
    }

    if ($html !== false && is_valid_bincheck_response($html)) {
        $binVal = $bin;
        $cardBrand = $cardType = $cardLevel = $issuerName = '';
        if (preg_match('/<td[^>]*>BIN\/IIN<\/td>\s*<td[^>]*>(\d+)<\/td>/s', $html, $m)) $binVal = trim($m[1]);
        if (preg_match('/<td[^>]*>Card Brand<\/td>\s*<td[^>]*>([^<]+)<\/td>/s', $html, $m)) $cardBrand = trim(strip_tags($m[1]));
        if (preg_match('/<td[^>]*>Card Type<\/td>\s*<td[^>]*>([^<]+)<\/td>/s', $html, $m)) $cardType = trim(strip_tags($m[1]));
        if (preg_match('/<td[^>]*>Card Level<\/td>\s*<td[^>]*>([^<]+)<\/td>/s', $html, $m)) $cardLevel = trim(strip_tags($m[1]));
        if (preg_match('/<td[^>]*>Issuer Name \/ Bank<\/td>\s*<td[^>]*>(.*?)<\/td>/s', $html, $m)) $issuerName = trim(strip_tags($m[1]));
        if (empty($cardBrand) && preg_match('/is a valid BIN number ([A-Z\s]+) issued by ([^.]+) in/i', $html, $m)) {
            $cardBrand = trim($m[1]);
            if (empty($issuerName)) $issuerName = trim($m[2]);
        }
        $parts = array_filter([$cardBrand, $cardType, $cardLevel, $issuerName]);
        return 'Bin            :: ' . $binVal . ' ' . (empty($parts) ? '(data tidak ditemukan)' : implode(' ', $parts));
    }

    // Fallback binlist.net
    $ch = curl_init('https://lookup.binlist.net/' . $bin);
    curl_setopt_array($ch, [
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_TIMEOUT => 8,
        CURLOPT_CONNECTTIMEOUT => 5,
        CURLOPT_HTTPHEADER => ['Accept-Version: 3'],
        CURLOPT_USERAGENT => 'Mozilla/5.0 (compatible; BINChecker/1.0)',
    ]);
    $json = curl_exec($ch);
    curl_close($ch);
    if ($json !== false) {
        $data = @json_decode($json, true);
        if (!empty($data['scheme']) || !empty($data['bank']['name'])) {
            $cardBrand = isset($data['scheme']) ? strtoupper($data['scheme']) : '';
            $cardType = isset($data['type']) ? strtoupper($data['type']) : '';
            $cardLevel = (isset($data['brand']) && stripos($data['brand'], 'enhanced') !== false) ? 'ENHANCED' : '';
            if (empty($cardLevel) && !empty($data['brand']) && stripos($data['brand'], 'standard') !== false) $cardLevel = 'STANDARD';
            $issuerName = isset($data['bank']['name']) ? trim($data['bank']['name']) : '';
            $parts = array_filter([$cardBrand, $cardType, $cardLevel, $issuerName]);
            return 'Bin            :: ' . $bin . ' ' . (empty($parts) ? '(data tidak ditemukan)' : implode(' ', $parts));
        }
    }
    return 'Bin            :: ' . $bin . ' (gagal)';
}

// --- Web UI ---
$run = isset($_GET['run']) || isset($_POST['run']);

if (!$run) {
    $listExists = is_file($LIST_FILE);
    $listPreview = $listExists ? trim(file_get_contents($LIST_FILE)) : '';
    $listCount = $listExists ? count(array_filter(explode("\n", $listPreview), function ($l) {
        return strlen(substr(preg_replace('/\D/', '', trim($l)), 0, 6)) >= 6;
    })) : 0;
    ?>
<!DOCTYPE html>
<html lang="id">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Mass BIN Check</title>
    <style>
        * { box-sizing: border-box; }
        body { font-family: system-ui, sans-serif; max-width: 560px; margin: 2rem auto; padding: 0 1rem; background: #1a1a2e; color: #eee; }
        h1 { font-size: 1.35rem; margin-bottom: 0.5rem; }
        p { color: #aaa; font-size: 0.95rem; margin: 0.5rem 0; }
        .box { background: #16213e; border-radius: 8px; padding: 1.25rem; margin: 1rem 0; }
        .btn { display: inline-block; background: #0f3460; color: #fff; padding: 0.6rem 1.2rem; border-radius: 6px; text-decoration: none; border: none; cursor: pointer; font-size: 1rem; }
        .btn:hover { background: #1a4a7a; }
        .btn:disabled { opacity: 0.6; cursor: not-allowed; }
        .muted { color: #6c757d; font-size: 0.85rem; }
        ul { margin: 0.5rem 0; padding-left: 1.2rem; color: #aaa; }
        a { color: #7eb8da; }
    </style>
</head>
<body>
    <h1>Mass BIN Check</h1>
    <p>Baca dari <strong>list.txt</strong> (satu BIN atau nomor kartu per baris). Jika 16 digit, otomatis diambil 6 digit pertama. Hasil ke <strong>result.txt</strong>.</p>
    <div class="box">
        <?php if (!$listExists): ?>
            <p>File <code>list.txt</code> belum ada. Buat file tersebut di folder ini, isi satu BIN (6 digit) atau nomor kartu (16 digit) per baris.</p>
        <?php else: ?>
            <p>Ditemukan <strong><?= $listCount ?></strong> baris yang valid (min. 6 digit).</p>
            <form method="get" action="">
                <input type="hidden" name="run" value="1">
                <button type="submit" class="btn">Start Mass Check</button>
            </form>
        <?php endif; ?>
    </div>
    <p class="muted">Format list.txt: satu nomor per baris (6 digit BIN atau 16 digit kartu). Hasil: result.txt.</p>
</body>
</html>
    <?php
    exit;
}

// --- Run mass check ---
header('Content-Type: text/html; charset=utf-8');

if (!function_exists('curl_init')) {
    echo '<p>Error: cURL tidak tersedia.</p>';
    exit;
}

if (!is_file($LIST_FILE)) {
    echo '<p>File list.txt tidak ditemukan.</p><p><a href="">Kembali</a></p>';
    exit;
}

$lines = file($LIST_FILE, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
$bins = [];
foreach ($lines as $line) {
    $digits = preg_replace('/\D/', '', trim($line));
    if (strlen($digits) >= 6) {
        $bins[] = substr($digits, 0, 6);
    }
}
$bins = array_values(array_unique($bins));
$total = count($bins);

if ($total === 0) {
    echo '<p>Tidak ada BIN valid di list.txt (min. 6 digit per baris).</p><p><a href="">Kembali</a></p>';
    exit;
}

$resultLines = [];
echo '<pre style="background:#16213e;padding:1rem;border-radius:8px;color:#eee;">';
echo "Memeriksa {$total} BIN...\n\n";
if (ob_get_level()) ob_end_flush();
flush();

for ($i = 0; $i < $total; $i++) {
    $bin = $bins[$i];
    $num = $i + 1;
    echo "({$num}/{$total}) {$bin} ... ";
    flush();
    $line = get_bin_line($bin);
    $resultLines[] = $line ?: ('Bin            :: ' . $bin . ' (gagal)');
    echo ($line && strpos($line, '(gagal)') === false ? "OK" : "gagal") . "\n";
    flush();
    if ($i < $total - 1) sleep($DELAY_SEC);
}

$written = file_put_contents($RESULT_FILE, implode("\n", $resultLines) . "\n", LOCK_EX);
echo "\nSelesai. " . count($resultLines) . " baris ditulis ke result.txt.\n";
echo '</pre>';
echo '<p><a href="result.txt" class="btn" download>Download result.txt</a> &nbsp; <a href="">Kembali</a></p>';
echo '<p class="muted">Atau buka: <a href="result.txt">result.txt</a></p>';
?>
